<?php
/**
 * Core: The Oxyprops class
 *
 * @link        https://oxyprops.com
 *
 * @package     OxyProps
 * @subpackage  Core
 * @author      Cédric Bontems <cedric@thewebforge.dev>
 * @since       1.5.0
 * @copyright   Copyright (c) 2022, Cédric Bontems
 * @license     https://www.gnu.org/licenses/gpl-2.0.html  GPLv2 or later
 */

namespace OxyProps\Inc;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * OxyProps\Inc\Oxyprops class.
 *
 * Core class used to organize plugin related data.
 *
 * @since 1.5.0 Creation.
 * @author Cédric Bontems <cedric@thewebforge.dev>
 */
class Oxyprops {


	use Singleton;

	/**
	 * The plugin icon as a base64 encoded string of text.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $icon Icon as a base64 SVG image.
	 */
	private string $icon;

	/**
	 * The system path to the plugin directory.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $path Full system path to the plugin directory with trailing slash.
	 */
	private string $path;

	/**
	 * The plugin slug.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $slug The OxyProps plugin slug.
	 */
	private string $slug;

	/**
	 * The system path to the plugin file.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $file Full system path to the OxyProps plugin php file.
	 */
	private string $file;

	/**
	 * The full URL to the plugin folder.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $url Full URL to OxyProps plugin folder with trailing slash.
	 */
	private string $url;

	/**
	 * The relative path to the plugin file from WP Plugins folder.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $plugin Relative path to the plugin php file.
	 */
	private string $plugin;

	/**
	 * The plugin name.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $name The plugin name retreived from the plugin file meta data.
	 */
	private string $name;

	/**
	 * The plugin version.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $version The plugin version retreived from the plugin file meta data.
	 */
	private string $version;

	/**
	 * The plugin Text Domain.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var string $textdomain The plugin text domain retreived from the plugin file meta data.
	 */
	private string $textdomain;

	/**
	 * The plugin options.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var array $current_settings The plugin current settings retreived from the database.
	 */
	private array $current_settings;

	/**
	 * Initialize Oxyprops class.
	 *
	 * Called by the constructor. It retreives and populates all the class properties.
	 *
	 * @see https://developer.wordpress.org/reference/functions/get_plugin_data/ get_plugin_data()
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	private function register(): void {

		// retreives the function from wp-admin if needed.
		if ( ! function_exists( 'get_plugin_data' ) ) {
			include_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$this->set_icon( 'data:image/svg+xml;base64,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' );
		$this->set_path( plugin_dir_path( __DIR__ ) );
		$this->set_slug( plugin_basename( dirname( __DIR__ ) ) );
		$this->set_file( $this->path . $this->slug . '.php' );
		$this->set_url( plugin_dir_url( dirname( __FILE__ ) ) );
		$this->set_plugin( $this->slug . '/' . $this->slug . '.php' );

		$plugin_data = get_plugin_data( $this->file, false, false );

		$this->set_name( $plugin_data['Name'] );
		$this->set_textdomain( $plugin_data['TextDomain'] );
		$this->set_version( $plugin_data['Version'] );
		$this->set_current_settings( $this->retreive_plugin_current_settings() );
	}

	/**
	 * Reads current settings from the database.
	 *
	 * It returns the current option value if it exist in the database, otherwise,
	 * it returns the default settings.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return array The active settings.
	 */
	private function retreive_plugin_current_settings(): array {
		$defaults = Init::get_instance( 'defaults' );
		if ( method_exists( $defaults, 'get_oxyprops_default_settings' ) ) {
			return get_option( $this->slug, array( 'settings' => $defaults->get_oxyprops_default_settings() ) );
		}
	}

	// * Getters and Setters.

	/**
	 * Get the main plugin file.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string Full system path to the plugin file.
	 */
	public function get_file(): string {
		return $this->file;
	}

	/**
	 * Set the main plugin file.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $file Provide a full system path to the plugin file.
	 * @return object The Oxyprops instance.
	 */
	public function set_file( string $file ): object {
		$this->file = $file;

		return $this;
	}

	/**
	 * Get the plugin SVG admin icon.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string The plugin admin icon.
	 */
	public function get_icon(): string {
		return $this->icon;
	}

	/**
	 * Set the plugin SVG admin icon.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $icon The icon to be used.
	 * @return object The Oxyprops instance.
	 */
	public function set_icon( string $icon ): object {
		$this->icon = $icon;

		return $this;
	}

	/**
	 * Get system path to the plugin folder.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string Full system path to the plugin folder.
	 */
	public function get_path(): string {
		return $this->path;
	}

	/**
	 * Set system path to the plugin folder.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $path Provide a full system path to the plugin folder.
	 * @return object The Oxyprops instance.
	 */
	public function set_path( string $path ): object {
		$this->path = $path;

		return $this;
	}

	/**
	 * Get the plugin slug.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string The plugin slug.
	 */
	public function get_slug(): string {
		return $this->slug;
	}

	/**
	 * Set the plugin slug.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $slug Provide the plugin slug.
	 * @return object The oxyprops instance.
	 */
	public function set_slug( string $slug ): object {
		$this->slug = $slug;

		return $this;
	}

	/**
	 * Get the URL to the plugin folder.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string The URL to the plugin folder.
	 */
	public function get_url(): string {
		return $this->url;
	}

	/**
	 * Set the URL to the plugin folder.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $url Provide the URL to the plugin folder.
	 * @return object The oxyprops instance.
	 */
	public function set_url( string $url ): object {
		$this->url = $url;

		return $this;
	}

	/**
	 * Get the path to the plugin file, relative to the plugins directory.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string the path to the plugin file, relative to the plugins directory.
	 */
	public function get_plugin(): string {
		return $this->plugin;
	}

	/**
	 * Set the path to the plugin file, relative to the plugins directory.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $plugin Provide the path to the plugin file, relative to the plugins directory.
	 * @return object The oxyprops instance.
	 */
	public function set_plugin( string $plugin ): object {
		$this->plugin = $plugin;

		return $this;
	}

	/**
	 * Get the plugin name.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string The plugin name.
	 */
	public function get_name(): string {
		return $this->name;
	}

	/**
	 * Set the plugin name.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $name Provide the plugin name.
	 * @return object The oxyprops instance.
	 */
	public function set_name( string $name ): object {
		$this->name = $name;

		return $this;
	}

	/**
	 * Get the plugin current version.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string The plugin current version.
	 */
	public function get_version(): string {
		return $this->version;
	}

	/**
	 * Set the plugin current version.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $version Provide the plugin current version.
	 * @return object The oxyprops instance.
	 */
	public function set_version( string $version ): object {
		$this->version = $version;

		return $this;
	}

	/**
	 * Get the plugin textdomain.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return string The plugin textdomain.
	 */
	public function get_textdomain(): string {
		return $this->textdomain;
	}

	/**
	 * Summary.
	 *
	 * Description.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param string $textdomain Provide the plugin textdomain.
	 * @return object The oxyprops instance.
	 */
	public function set_textdomain( string $textdomain ): object {
		$this->textdomain = $textdomain;

		return $this;
	}

	/**
	 * Get the plugin current settings.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return array The plugin current settings.
	 */
	public function get_current_settings(): array {
		return $this->current_settings;
	}

	/**
	 * Set the plugin current settings.
	 *
	 * @since 1.5.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @param array $current_settings The plugin current settings.
	 * @return object The oxyprops instance.
	 */
	public function set_current_settings( array $current_settings ): object {
		$this->current_settings = $current_settings;

		return $this;
	}
}
